# Simulate coalescent process with sampling for use in model selection

# Assumptions and modifications
# - single trajectories simulated
# - samples placed uniformly over cycles or segments

# Clean the workspace and console
closeAllConnections()
rm(list=ls())
cat("\014")  
graphics.off()

# Packages for phylodyn
library("sp")
library("INLA")
library("spam")
library("ape")
library("devtools")
library("phylodyn")

# Set working directory to source
this.dir <- dirname(parent.frame(2)$ofile)
setwd(this.dir)

# Function to write simple csv files to correct path
tableWrite <- function(val, name, pathname) {
  # Add path to name
  str0 <- paste(c(pathname, name), collapse = "")
  # Write table
  write.table(val, str0, row.names=FALSE, col.names=FALSE, sep=",")
}

# Demographic functions considered --------------------------------------------------------------------

# Define a logistic trajectory with larger N
N = 10^3; N0 = 0.01*N
logistic_traj2 <- function (t, offset = 0, a = 2) 
{
  t = t + offset
  result = rep(0, length(t))
  result[(t%%12) <= 6] = N0 + N/(1 + exp((3 - (t[(t%%12) <= 6]%%12)) * a))
  result[(t%%12) > 6] = N0 + N/(1 + exp(((t[(t%%12) > 6]%%12) - 12 + 3) * a))
  return(result)
}
# Define a boom-bust with a later changepoint and an offset
boombust_traj3 <- function (t, bust = 10, scale = 1000, offset = 100) 
{
  result = rep(0, length(t))
  result[t <= bust] = scale*exp(t[t <= bust] - bust) + offset
  result[t > bust] = scale*exp(bust - t[t > bust]) + offset
  return(result)
}

# Define larger exponential
exp_traj2 <- function (t, scale = 10000, rate = 1) 
{
  return(scale*exp(-t*rate))
}
# Define a middling bottleneck
bottleneck_traj2 <- function (t) 
{
  result = rep(0, length(t))
  result[t <= 15] <- 500
  result[t > 15 & t < 40] <- 20
  result[t >= 40] <- 500
  return(result)
}
# Low and high constant populations
unif_traj_low <- function (t, level = 50) 
{
  n = length(t)
  return(rep(level, n))
}
unif_traj_high <- function (t, level = 5000) 
{
  n = length(t)
  return(rep(level, n))
}

# Define binary rate-shift with population fall
tshift = 400
shift_traj <- function (t, frac = fracs[i], ts = tshift) 
{
  result = rep(0, length(t))
  result[t <= tshift] <- 1000
  result[t >= tshift] <- 0.5*1000
  return(result)
}

# Main code for preferential simulations --------------------------------------------------------------------

# Possible trajectories
trajNames = c('logis', 'exp', 'steep', 'unif_low', 'unif_high', 'boom', 'cyc', 'bottle', 'mesa', 'binary')
expTrajs = c(2, 4, 5, 6) # non-cyclic trajectories
num = length(trajNames)

for (type in 1:num) {
  # Set population true trajectory
  trajName = trajNames[type]
  # Choose trajectory type
  trajType = switch(type,
                    "1"= logistic_traj2,
                    "2"= exp_traj2,
                    "3"= steep_cyc_traj,
                    "4"= unif_traj_low,
                    "5"= unif_traj_high,
                    "6"= boombust_traj3,
                    "7"= cyclic_traj,
                    "8"= bottleneck_traj2,
                    "9"= mesa_traj,
                    "10" = shift_traj
  )
  traj = trajType
  
  # Create folder for traj specific results
  trajNameSplit = paste(c(trajName, '_test'), collapse = '')
  dir.create(file.path(this.dir, trajNameSplit))
  
  # Set sampling interval end and no. samples
  if(!is.element(type, expTrajs)){
    # Cyclic trajectories sampled evenly
    all_samp_end = 48
    nsamps = 801; ndivs = 20
    # Sample number and times
    samps = c(rep(floor(nsamps/ndivs), ndivs-1), nsamps-(ndivs-1)*floor(nsamps/ndivs))
    samp_times = seq(0, all_samp_end, length.out = ndivs)
  } else{
    # Isochronous sampling of non-cyclic
    samps = 401
    samp_times = 0
  }
  
  # Simulate genealogy and get all times
  gene = coalsim(samp_times = samp_times, n_sampled = samps, traj = traj, lower_bound = 10, method = "thin")
  coal_times = gene$coal_times
  coalLin = gene$lineages
  
  # Obtain true trajectory across time
  tmax = max(coal_times)
  t = seq(0, tmax, length=40000)
  y = traj(t)
  
  # Export data for Matlab
  pathf = paste(c(this.dir, '/', trajNameSplit, '/'), collapse = "")
  tableWrite(coal_times, 'coaltimes.csv', pathf)
  tableWrite(samp_times, 'samptimes.csv', pathf)
  tableWrite(coalLin, 'coalLin.csv', pathf)
  tableWrite(y, 'trajy.csv', pathf)
  tableWrite(t, 'trajt.csv', pathf)
  tableWrite(samps, 'sampIntro.csv', pathf)
  
  # Plot and write tree
  tree <-generate_newick(gene)
  currDir = this.dir
  setwd(file.path(this.dir, trajNameSplit))
  write.tree(tree$newick, file="tree.txt")
  setwd(currDir)
}
